% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summary-index.R
\name{summary-index}
\alias{summary-index}
\alias{slide_index_sum}
\alias{slide_index_prod}
\alias{slide_index_mean}
\alias{slide_index_min}
\alias{slide_index_max}
\alias{slide_index_all}
\alias{slide_index_any}
\title{Specialized sliding functions relative to an index}
\usage{
slide_index_sum(
  x,
  i,
  ...,
  before = 0L,
  after = 0L,
  complete = FALSE,
  na_rm = FALSE
)

slide_index_prod(
  x,
  i,
  ...,
  before = 0L,
  after = 0L,
  complete = FALSE,
  na_rm = FALSE
)

slide_index_mean(
  x,
  i,
  ...,
  before = 0L,
  after = 0L,
  complete = FALSE,
  na_rm = FALSE
)

slide_index_min(
  x,
  i,
  ...,
  before = 0L,
  after = 0L,
  complete = FALSE,
  na_rm = FALSE
)

slide_index_max(
  x,
  i,
  ...,
  before = 0L,
  after = 0L,
  complete = FALSE,
  na_rm = FALSE
)

slide_index_all(
  x,
  i,
  ...,
  before = 0L,
  after = 0L,
  complete = FALSE,
  na_rm = FALSE
)

slide_index_any(
  x,
  i,
  ...,
  before = 0L,
  after = 0L,
  complete = FALSE,
  na_rm = FALSE
)
}
\arguments{
\item{x}{\verb{[vector]}

A vector to compute the sliding function on.
\itemize{
\item For sliding sum, mean, prod, min, and max, \code{x} will be cast to a double
vector with \code{\link[vctrs:vec_cast]{vctrs::vec_cast()}}.
\item For sliding any and all, \code{x} will be cast to a logical vector with
\code{\link[vctrs:vec_cast]{vctrs::vec_cast()}}.
}}

\item{i}{\verb{[vector]}

The index vector that determines the window sizes. It is fairly common to
supply a date vector as the index, but not required.

There are 3 restrictions on the index:
\itemize{
\item The size of the index must match the size of \code{.x}, they will not be
recycled to their common size.
\item The index must be an \emph{increasing} vector, but duplicate values
are allowed.
\item The index cannot have missing values.
}}

\item{...}{These dots are for future extensions and must be empty.}

\item{before, after}{\verb{[vector(1) / function / Inf]}
\itemize{
\item If a vector of size 1, these represent the number of values before or
after the current element of \code{.i} to include in the sliding window.
Negative values are allowed, which allows you to "look forward" from the
current element if used as the \code{.before} value, or "look backwards" if used
as \code{.after}. Boundaries are computed from these elements as \code{.i - .before}
and \code{.i + .after}. Any object that can be added or subtracted from \code{.i}
with \code{+} and \code{-} can be used. For example, a lubridate period, such as
\code{\link[lubridate:period]{lubridate::weeks()}}.
\item If \code{Inf}, this selects all elements before or after the current element.
\item If a function, or a one-sided formula which can be coerced to a function,
it is applied to \code{.i} to compute the boundaries. Note that this function
will only be applied to the \emph{unique} values of \code{.i}, so it should not rely
on the original length of \code{.i} in any way. This is useful for applying a
complex arithmetic operation that can't be expressed with a single \code{-} or
\code{+} operation. One example would be to use \code{\link[lubridate:mplus]{lubridate::add_with_rollback()}}
to avoid invalid dates at the end of the month.
}

The ranges that result from applying \code{.before} and \code{.after} have the same
3 restrictions as \code{.i} itself.}

\item{complete}{\verb{[logical(1)]}

Should the function be evaluated on complete windows only? If \code{FALSE},
the default, then partial computations will be allowed.}

\item{na_rm}{\verb{[logical(1)]}

Should missing values be removed from the computation?}
}
\value{
A vector the same size as \code{x} containing the result of applying the
summary function over the sliding windows.
\itemize{
\item For sliding sum, mean, prod, min, and max, a double vector will be
returned.
\item For sliding any and all, a logical vector will be returned.
}
}
\description{
These functions are specialized variants of the most common ways that
\code{\link[=slide_index]{slide_index()}} is generally used. Notably, \code{\link[=slide_index_sum]{slide_index_sum()}} can be used
for rolling sums relative to an index (like a Date column), and
\code{\link[=slide_index_mean]{slide_index_mean()}} can be used for rolling averages.

These specialized variants are \emph{much} faster and more memory efficient than
using an otherwise equivalent call constructed with \code{\link[=slide_index_dbl]{slide_index_dbl()}}
or \code{\link[=slide_index_lgl]{slide_index_lgl()}}, especially with a very wide window.
}
\details{
For more details about the implementation, see the help page of
\code{\link[=slide_sum]{slide_sum()}}.
}
\examples{
x <- c(1, 5, 3, 2, 6, 10)
i <- as.Date("2019-01-01") + c(0, 1, 3, 4, 6, 8)

# `slide_index_sum()` can be used for rolling sums relative to an index,
# allowing you to "respect gaps" in your series. Notice that the rolling
# sum in row 3 is only computed from `2019-01-04` and `2019-01-02` since
# `2019-01-01` is more than two days before the current date.
data.frame(
  i = i,
  x = x,
  roll = slide_index_sum(x, i, before = 2)
)

# `slide_index_mean()` can be used for rolling averages
slide_index_mean(x, i, before = 2)

# Only evaluate the sum on windows that have the potential to be complete
slide_index_sum(x, i, before = 2, after = 1, complete = TRUE)
}
\seealso{
\code{\link[=slide_sum]{slide_sum()}}
}
