// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef MEDIA_AUDIO_CRAS_AUDIO_MANAGER_CRAS_H_
#define MEDIA_AUDIO_CRAS_AUDIO_MANAGER_CRAS_H_

#include <cras_types.h>

#include <memory>
#include <string>
#include <vector>

#include "base/compiler_specific.h"
#include "base/macros.h"
#include "base/memory/ref_counted.h"
#include "media/audio/cras/audio_manager_cras_base.h"

namespace media {

class MEDIA_EXPORT AudioManagerCras : public AudioManagerCrasBase {
 public:
  AudioManagerCras(std::unique_ptr<AudioThread> audio_thread,
                   AudioLogFactory* audio_log_factory);
  ~AudioManagerCras() override;

  // AudioManager implementation.
  bool HasAudioOutputDevices() override;
  bool HasAudioInputDevices() override;
  void GetAudioInputDeviceNames(AudioDeviceNames* device_names) override;
  void GetAudioOutputDeviceNames(AudioDeviceNames* device_names) override;
  AudioParameters GetInputStreamParameters(
      const std::string& device_id) override;
  std::string GetDefaultInputDeviceID() override;
  std::string GetDefaultOutputDeviceID() override;

  // AudioManagerCras implementation.
  bool IsDefault(const std::string& device_id, bool is_input) override;

 protected:
  AudioParameters GetPreferredOutputStreamParameters(
      const std::string& output_device_id,
      const AudioParameters& input_params) override;

 private:
  uint64_t GetPrimaryActiveInputNode();
  uint64_t GetPrimaryActiveOutputNode();
  void GetPrimaryActiveInputNodeOnMainThread(uint64_t* active_input_node_id,
                                             base::WaitableEvent* event);
  void GetPrimaryActiveOutputNodeOnMainThread(uint64_t* active_output_node_id,
                                              base::WaitableEvent* event);
  void GetDefaultOutputBufferSizeOnMainThread(int32_t* buffer_size,
                                              base::WaitableEvent* event);

  // Task runner of browser main thread. CrasAudioHandler should be only
  // accessed on this thread.
  scoped_refptr<base::SingleThreadTaskRunner> main_task_runner_;

  // For posting tasks from audio thread to |main_task_runner_|.
  base::WeakPtr<AudioManagerCras> weak_this_;

  base::WeakPtrFactory<AudioManagerCras> weak_ptr_factory_;

  DISALLOW_COPY_AND_ASSIGN(AudioManagerCras);
};

}  // namespace media

#endif  // MEDIA_AUDIO_CRAS_AUDIO_MANAGER_CRAS_H_
