%feature("docstring") OT::GaussLegendre
R"RAW(Tensorized integration algorithm of Gauss-Legendre.

Available constructors:
    GaussLegendre(*dimension=1*)

    GaussLegendre(*discretization*)

Parameters
----------
dimension : int, :math:`dimension>0`
    The input dimension of the functions to integrate.
discretization : sequence of int
    The number of nodes in each dimension. The sequence must be non-empty and must contain only positive values.
    
    The default discretization is stored as the `GaussLegendre-
    DefaultMarginalIntegrationPointsNumber` :class:`~openturns.ResourceMap` key,
    which is used for each dimension.

Notes
-----
We consider a function :math:`f: \Rset^\inputDim \mapsto \Rset^\outputDim` and a domain :math:`\set{D} = [\vect{a}, \vect{b}]` where :math:`\vect{a}, \vect{b} \in \Rset^\inputDim`.
The Gauss-Legendre algorithm approximates the definite integral:

.. math::
    :label: intInit

    \int_{\vect{a}}^\vect{b} f(\vect{t})\di{\vect{t}}


For :math:`i \in \{1, ..., \inputDim\}`, let :math:`N_i \in \Nset` be the number
of marginal nodes of the univariate Gauss-Legendre quadrature.
The algorithm uses a fixed tensorized Gauss-Legendre approximation based
on the tensor-product Gauss quadrature using
:math:`N_1, ..., N_\inputDim \in \Nset` marginal nodes.
In other words, this algorithm uses an anisotropic quadrature rule,
i.e. the number of nodes in each marginal dimension
are not necessarily equal.

In order to get a more straightforward access to the nodes and weights,
please use the :class:`~openturns.GaussProductExperiment` class.
In order to create a more general tensor product quadrature,
please use the :class:`~openturns.TensorProductExperiment` class.

**Quadrature rule**

We detail here the quadrature rule. As integral :eq:`intInit` is computed for each marginal of the integrand function,
we consider now that :math:`f: \Rset^\inputDim \mapsto \Rset`. The approximated integral is:

.. math::
    :label: approxInt

    \int_{\vect{a}}^{\vect{b}} f(\vect{t}) \di{\vect{t}} = \sum_{n_1=1}^{N_1} \dots  \sum_{n_d=1}^{N_d}\left( \prod_{i=1}^d w^{N_i}_{n_i}\right) f(\xi^{N_1}_{n_1},\dots,\xi^{N_d}_{n_d})

where :math:`\xi^{N_i}_{n_i}` is the :math:`n_i` -th node computed from the :math:`N_i` -points Gauss-
Legendre 1D integration rule and :math:`w^{N_i}_{n_i} > 0` the associated weight.

The total number of nodes is the product of marginal number of nodes:

.. math::

    \prod_{i = 1}^\inputDim N_i.

To use that approximation, the integration variables are scaled to the domain :math:`[0,1]^d` with the
mapping function :math:`\varphi` defined by:

.. math::
    :label: mapping

    \varphi : & [0,1]^d  \rightarrow [\vect{a}, \vect{b}]\\
           & \vect{x} \rightarrow \vect{t} = \vect{a} + (\vect{b}-\vect{a}) \vect{x}

where each component is :math:`t_i = a_i + (b_i-a_i)x_i` for :math:`1 \leq i \leq d`.

Let :math:`|\vect{b}-\vect{a}| = \prod_{i=1}^d (b_i-a_i)` be the volume of the domain :math:`[\vect{a},
\vect{b}]`. Then, we have:

.. math::
    :label: equivInt

    \int_{\vect{a}}^{\vect{b}} f(\vect{t})\di{\vect{t}} & = \int_{[0,1]^d} |\vect{b}-\vect{a}| f\left(
    \vect{a} + (\vect{b}-\vect{a})\vect{x}\right)\di{\vect{x}} \\
    & = \int_{[0,1]^d} \tilde{f}(\vect{x})\di{\vect{x}}

where we introduced the scaled function :math:`\tilde{f}` on :math:`[0,1]^d` defined by:

.. math::
    :label: scaledFunc

    \tilde{f}(\vect{x}) = |\vect{b}-\vect{a}| f \circ \varphi ( \vect{x} )

Then, the Gauss-Legendre quadrature rule is applied to the function :math:`\tilde{f}` defined on
:math:`[0,1]^d`. We denote by :math:`\tilde{\xi}^{N_i}_{n_i}` the :math:`n_i` -th node of the :math:`N_i` 
-points Gauss-Legendre 1D integration rule and :math:`\tilde{w}^{N_i}_{n_i}` the associated weight.
Then we have:

.. math::
    :label: approxIntStandard

    \int_{[0,1]^d} \tilde{f}(\vect{x})\di{\vect{x}} = \sum_{n_1=1}^{N_1}\dots  \sum_{n_d=1}^{N_d}
    \left(\prod_{i=1}^d \tilde{w}^{N_i}_{n_i}\right)\tilde{f}(\tilde{\xi}^{N_1}_{n_1},\dots,\tilde{\xi}^{N_d}
    _{n_d})

Using :eq:`scaledFunc` and :eq:`approxIntStandard`, we get:

.. math::
    :label: intiInt2

    \int_{\vect{a}}^{\vect{b}} f(\vect{t})\di{\vect{t}} & = \sum_{n_1=1}^{N_1}\dots  \sum_{n_d=1}^{N_d}
    \left(\prod_{i=1}^d \tilde{w}^{N_i}_{n_i}\right) |\vect{b}-\vect{a}| f\left( a_1 + (b_1-a_1)\tilde{\xi}
    ^{N_1}_{n_1},\dots,a_d + (b_d-a_d)\tilde{\xi}^{N_d}_{n_d}\right)\\
    & =  \sum_{n_1=1}^{N_1}\dots  \sum_{n_d=1}^{N_d}\left(\prod_{i=1}^d \tilde{w}^{N_i}_{n_i}(b_i-a_i)\right)
    f\left( a_1 + (b_1-a_1)\tilde{\xi}^{N_1}_{n_1},\dots,a_d + (b_d-a_d)\tilde{\xi}^{N_d}_{n_d}\right)
    
Then, the nodes used to compute approximation :eq:`approxInt` are defined as follows: 

.. math::
    :label: nodesIntInit

    \xi^{N_i}_{n_i} = a_i + (b_i-a_i)\tilde{\xi}^{N_i}_{n_i}

and the weights are: 

.. math::
    :label: weightsIntInit

    w^{N_i}_{n_i} = (b_i-a_i)\tilde{w}^{N_i}_{n_i}


**Polynomial exactness**

The Gauss-Legendre quadrature rule is exact for polynomials
up to some degree.
More precisely, for any :math:`m_i \in \Nset`, let :math:`\mathcal{P}_{m_i}^{(1)}`
be the set of mono-variable polynomials of degree lower or equal to :math:`m_i`.
Consider the tensor product of 1D polynomials:

.. math::

    \bigotimes_{i = 1}^\inputDim \mathcal{P}_{m_i}^{(1)}
    = 
    \left\{
    (x_1, ..., x_\inputDim)\in\Rset^\inputDim
    \rightarrow \prod_{i = 1}^\inputDim p_i(x_i) \in \Rset, \quad 
    p_i \in \mathcal{P}_{m_i}^{(1)}
    \right\}.

Therefore the Gauss-Legendre tensorized quadrature is exact for all
polynomials of the vector space:

.. math::

    \bigotimes_{i = 1}^\inputDim \mathcal{P}_{2 n_i - 1}^{(1)}.

See also
--------
GaussProductExperiment, TensorProductExperiment


    
Examples
--------
Create a Gauss-Legendre algorithm in dimension 2.

>>> import openturns as ot
>>> algo = ot.GaussLegendre(2)

Create a Gauss-Legendre algorithm in dimension 3 with 2 nodes for :math:`x_1`,
4 nodes for :math:`x_2` and 5 nodes for :math:`x_3`.

>>> algo = ot.GaussLegendre([2, 4, 5])

We show how to use this method to evaluate the following integral:

.. math::

    \int_{[0,1]^3} x_1^5 x_2^3 x_3^7 d\vect{x}

First, we use the default number of nodes as defined in the
:class:`~openturns.ResourceMap`.

>>> dimension = 3
>>> bounds = ot.Interval([0.0] * dimension, [1.0] * dimension)
>>> polynomial = ot.SymbolicFunction(['x1', 'x2', 'x3'], ['x1^5 * x2^3 * x3^7'])
>>> algo = ot.GaussLegendre(dimension)
>>> computedIntegral = algo.integrate(polynomial, bounds)
>>> print(computedIntegral)
[0.00520833]

In the previous example, we used the default number of nodes in each dimension,
as defined by the :class:`~openturns.ResourceMap`.
In the next example, we set the number of marginal nodes:
we use 3 nodes for :math:`x_1`, 2 nodes for :math:`x_2`, 4 nodes for :math:`x_3`.
In this case, the vector space for which the Gauss-Legendre quadrature
rule is exact is :math:`\mathcal{P}_5 \otimes \mathcal{P}_3 \otimes \mathcal{P}_7`.
In other words, we test the quadrature rule against the
polynomial with maximum possible marginal degrees.

>>> dimension = 3
>>> bounds = ot.Interval([0.0] * dimension, [1.0] * dimension)
>>> polynomial = ot.SymbolicFunction(['x1', 'x2', 'x3'], ['x1^5 * x2^3 * x3^7'])
>>> algo = ot.GaussLegendre([3, 2, 4])
>>> computedIntegral = algo.integrate(polynomial, bounds)
>>> print(computedIntegral)
[0.00520833])RAW"

// ---------------------------------------------------------------------
%feature("docstring") OT::GaussLegendre::integrate
R"RAW(Evaluation of the integral of :math:`f` on an interval.

Available usages:
    integrate(*f, interval*)

Parameters
----------
f : :class:`~openturns.Function`, :math:`f: \Rset^\inputDim \mapsto \Rset^\outputDim`
    The integrand function.
interval : :class:`~openturns.Interval`, :math:`interval \in \Rset^\inputDim` 
    The integration domain. 

Returns
-------
value : :class:`~openturns.Point`
    Approximation of the integral.


Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x'], ['sin(x)'])
>>> a = -2.5
>>> b = 4.5
>>> algoGL = ot.GaussLegendre([10])
>>> value = algoGL.integrate(f, ot.Interval(a, b))[0]
>>> print(value)
-0.590...
)RAW"

// ---------------------------------------------------------------------
%feature("docstring") OT::GaussLegendre::integrateWithNodes
R"RAW(Evaluation of the integral of :math:`f` on an interval with nodes.

Parameters
----------
f : :class:`~openturns.Function`, :math:`f: \Rset^\inputDim \mapsto \Rset^\outputDim`
    The integrand function.
interval : :class:`~openturns.Interval`, :math:`interval \in \Rset^\inputDim`
    The integration domain.

Returns
-------
value : :class:`~openturns.Point`
    Approximation of the integral.
nodes : :class:`~openturns.Sample`.
    The integration nodes defined in :eq:`nodesIntInit`.
    
Notes
-----
The nodes are those associated to the function: :math:`(\xi^{N_i}_{n_i})`.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x'], ['sin(x)'])
>>> a = -2.5
>>> b = 4.5
>>> algo = ot.GaussLegendre([10])
>>> value, nodes = algo.integrateWithNodes(f, ot.Interval(a, b))
)RAW"

// ---------------------------------------------------------------------
%feature("docstring") OT::GaussLegendre::getDiscretization
"Accessor to the discretization of the tensorized rule.

Returns
-------
discretization : :class:`~openturns.Indices`
    The number of integration point in each dimension."

// ---------------------------------------------------------------------
%feature("docstring") OT::GaussLegendre::getNodes
R"RAW(Accessor to the integration nodes in :math:`[0,1]^d`.

Returns
-------
nodes : :class:`~openturns.Sample`
    The tensorized Gauss-Legendre integration nodes on :math:`[0,1]^\inputDim`
    where :math:`d>0` is the dimension of the integration algorithm.
    
Notes
-----
The nodes are those associated to the scaled function: :math:`(\tilde{\xi}^{N_i}_{n_i})`.)RAW"

// ---------------------------------------------------------------------
%feature("docstring") OT::GaussLegendre::getWeights
R"RAW(Accessor to the integration weights for nodes in :math:`[0,1]^d`.

Returns
-------
weights : :class:`~openturns.Point`
    The tensorized Gauss-Legendre integration weights on :math:`[0,1]^\inputDim`
    where :math:`d>0` is the dimension of the integration algorithm.
    
Notes
-----
The weights are those associated to the scaled function: :math:`(\tilde{w}^{N_i}_{n_i})`.)RAW"

