/*
 *   Copyright (c) 1999-2001 Eric Gourgoulhon
 *
 *   This file is part of LORENE.
 *
 *   LORENE is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   LORENE is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with LORENE; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


char mat_cosp_legpp_C[] = "$Header: /cvsroot/Lorene/C++/Source/Non_class_members/Coef/mat_cosp_legpp.C,v 1.6 2014/10/13 08:53:13 j_novak Exp $" ;

/*
 * Fournit la matrice de passage pour la transformation des coefficients du
 * developpement en cos(2*j*theta) 
 * dans les coefficients du developpement en fonctions associees de Legendre 
 * P_l^m(cos(theta)) paires (i.e. telles que l-m est pair) avec m pair.
 * 
 * Cette routine n'effectue le calcul de la matrice que si celui-ci n'a pas 
 * deja ete fait, sinon elle renvoie le pointeur sur une valeur precedemment 
 * calculee.
 * 
 * Entree:
 * -------
 *  int np  :	Nombre de degres de liberte en phi 
 *  int nt  :	Nombre de degres de liberte en theta
 *
 * Sortie (valeur de retour) :
 * ---------------------------
 *  double* mat_cosp_legpp : pointeur sur le tableau contenant l'ensemble
 *			        (pour les np/2+1 valeurs de m: m=0,2,...,np) des 
 *				matrices de passage.
 *				La dimension du tableau est (np/2+1)*nt^2
 *				Le stokage est le suivant: 
 *
 *		mat_cosp_legpp[ nt*nt* m/2 + nt*l + j] = A_{mlj}
 *			    
 *				ou A_{mlj} est defini par
 *
 *   cos(2*j*theta) = som_{l=m/2}^{nt-1} A_{mlj} P_{2l}^m( cos(theta) )
 *
 *  ou P_n^m(x) represente la fonction de Legendre associee de degre n et 
 *     d'ordre m normalisee de facon a ce que
 *
 *	 int_0^pi [ P_n^m(cos(theta)) ]^2  sin(theta) dtheta = 1
 *
 * 
 */

/*
 * $Id: mat_cosp_legpp.C,v 1.6 2014/10/13 08:53:13 j_novak Exp $
 * $Log: mat_cosp_legpp.C,v $
 * Revision 1.6  2014/10/13 08:53:13  j_novak
 * Lorene classes and functions now belong to the namespace Lorene.
 *
 * Revision 1.5  2014/10/06 15:16:02  j_novak
 * Modified #include directives to use c++ syntax.
 *
 * Revision 1.4  2005/02/18 13:14:13  j_novak
 * Changing of malloc/free to new/delete + suppression of some unused variables
 * (trying to avoid compilation warnings).
 *
 * Revision 1.3  2003/01/31 10:31:24  e_gourgoulhon
 * Suppressed the directive #include <malloc.h> for malloc is defined
 * in <stdlib.h>
 *
 * Revision 1.2  2002/10/16 14:36:54  j_novak
 * Reorganization of #include instructions of standard C++, in order to
 * use experimental version 3 of gcc.
 *
 * Revision 1.1.1.1  2001/11/20 15:19:28  e_gourgoulhon
 * LORENE
 *
 * Revision 2.0  1999/02/22  15:35:27  hyc
 * *** empty log message ***
 *
 *
 *
 * $Header: /cvsroot/Lorene/C++/Source/Non_class_members/Coef/mat_cosp_legpp.C,v 1.6 2014/10/13 08:53:13 j_novak Exp $
 *
 */

// headers du C
#include <cstdlib>
#include <cmath>

// Prototypage
#include "headcpp.h"
#include "proto.h"

// Variable de loch
int loch_mat_cosp_legpp = 0 ;

namespace Lorene {
//******************************************************************************

double* mat_cosp_legpp(int np, int nt) {

#define NMAX	30		// Nombre maximun de couples(np,nt) differents 
static	double*	tab[NMAX] ;	// Tableau des pointeurs sur les tableaux 
static	int	nb_dejafait = 0 ;	// Nombre de tableaux deja initialises 
static	int	np_dejafait[NMAX] ;    // Valeurs de np pour lesquelles le
				       // calcul a deja ete fait
static	int	nt_dejafait[NMAX] ;    // Valeurs de np pour lesquelles le
				       // calcul a deja ete fait

int i, indice,  j,  j2,  m,  l ;
    
//    #pragma critical (loch_mat_cosp_legpp)
    {
	
    // Les matrices A_{mlj} pour ce couple (np,nt) ont-elles deja ete calculees ? 
    indice = -1 ;
    for ( i=0 ; i < nb_dejafait ; i++ ) {
	if ( (np_dejafait[i] == np) && (nt_dejafait[i] == nt) ) indice = i ;
    }


// Si le calcul n'a pas deja ete fait, il faut le faire : 
    if (indice == -1) {		   
	if ( nb_dejafait >= NMAX ) {
	    cout << "mat_cosp_legpp: nb_dejafait >= NMAX : "
		<< nb_dejafait << " <-> " << NMAX << endl ;
	    abort () ;	
	    exit(-1) ;
	    }
	indice = nb_dejafait ; 
	nb_dejafait++ ; 
	np_dejafait[indice] = np ;
	nt_dejafait[indice] = nt ;

	tab[indice] = new double[(np/2+1)*nt*nt] ;

//-----------------------
// Preparation du calcul 
//-----------------------

// Sur-echantillonnage pour calculer les produits scalaires sans aliasing:
	int nt2 = 2*nt - 1 ; 
	int nt2m1 = nt2 - 1 ; 

	int deg[3] ;
	deg[0] = 1 ;
	deg[1] = 1 ;
	deg[2] = nt2 ;

// Tableaux de travail
	double* yy = new double[nt2] ; 
	double* cost = new double[nt*nt2] ; 
   
// Calcul des cos(2*j*theta)  aux points de collocation
//  de l'echantillonnage double : 

	double dt = M_PI / double(2*(nt2-1)) ;
	for (j=0; j<nt; j++) {
	    for (j2=0; j2<nt2; j2++) {
		double theta = j2*dt ;
		cost[nt2*j + j2] = cos( 2*j * theta ) ;
	    }
	}	


//-------------------
// Boucle sur m
//-------------------

	int m_max = np ; 
	if (np == 1) m_max = 0 ; 

	for (m=0; m <= m_max ; m+=2) {

// Recherche des fonctions de Legendre associees d'ordre m :

	    double* leg = legendre_norm(m, nt) ;	
	
	    for (l=m/2; l<nt; l++) {	// boucle sur les P_{2l}^m
	    
		int ll = 2*l ;	// degre des fonctions de Legendre
		    
		for (j=0; j<nt; j++) {	// boucle sur les cos(2j theta)
			
//... produit scalaire de cos(2j theta) par P_{2l}^m(cos(theta)) 

		    for (j2=0; j2<nt2; j2++) {
			yy[nt2m1-j2] = cost[nt2*j + j2] *
					   leg[nt2* (ll-m) + j2] ;
		    }

//....... on passe en Tchebyshev vis-a-vis de x=cos(theta) pour	calculer
//	    l'integrale (routine int1d_chebp) : 		
		    cfrchebp(deg, deg, yy, deg, yy) ;
		    tab[indice][ nt*nt* m/2 + nt*l + j] = 
					    2.*int1d_chebp(nt2, yy) ;

		}	// fin de la boucle sur j  (indice de cos(2j theta) )
	    
	    }  // fin de la boucle sur l (indice de P_{2l}^m)
	    	    
	delete [] leg ; 
	   	    
	}  // fin de la boucle sur m

// Liberation espace memoire
// -------------------------

	delete [] yy ;
	delete [] cost ;

    } // fin du cas ou le calcul etait necessaire

    } //Fin de zone critique
    
    return tab[indice] ;

}


}
