/* 
 * Copyright (C) 2001-2013 Michael Fuchs
 *
 * This file is part of herold.
 * 
 * herold is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * herold is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with herold.  If not, see <http://www.gnu.org/licenses/>.  
 */
package org.dbdoclet.trafo.internal.html.docbook.editor;

import java.util.List;

import org.dbdoclet.tag.docbook.Abstract;
import org.dbdoclet.tag.docbook.Author;
import org.dbdoclet.tag.docbook.BaseTagFactory;
import org.dbdoclet.tag.docbook.Date;
import org.dbdoclet.tag.docbook.Info;
import org.dbdoclet.tag.docbook.Keyword;
import org.dbdoclet.tag.docbook.Keywordset;
import org.dbdoclet.tag.docbook.Para;
import org.dbdoclet.tag.docbook.Personname;
import org.dbdoclet.tag.docbook.Subject;
import org.dbdoclet.tag.docbook.Subjectset;
import org.dbdoclet.tag.docbook.Subjectterm;
import org.dbdoclet.tag.html.Meta;
import org.dbdoclet.trafo.html.EditorException;
import org.dbdoclet.trafo.html.EditorInstruction;
import org.dbdoclet.xiphias.dom.ElementImpl;
import org.dbdoclet.xiphias.dom.NodeImpl;

public class MetaEditor extends DocBookEditor {

	@Override
	public EditorInstruction edit(EditorInstruction values)
			throws EditorException {

		setValues(super.edit(values));
		BaseTagFactory dbfactory = getTagFactory();
		traverse(false);

		NodeImpl current = getCurrent();
		NodeImpl root = current.getRoot();
		ElementImpl info = (ElementImpl) root.getFirstChildElement();

		if (info == null) {
			info = dbfactory.createInfo();
			root.appendChild(info);
		}

		if (info instanceof Info == false) {
			info = dbfactory.createInfo();
			root.insertChild(0, info);
		}

		Meta meta = (Meta) getHtmlElement();
		String name = meta.getName();
		String content = meta.getContent();

		if (name != null && name.trim().equalsIgnoreCase("author")) {
			insertAuthor(dbfactory, info, content);
		}

		if (name != null && name.trim().equalsIgnoreCase("date")) {
			insertDate(dbfactory, info, content);
		}

		if (name != null && name.trim().equalsIgnoreCase("description")) {
			insertAbstract(dbfactory, info, content);
		}

		if (name != null && name.trim().equalsIgnoreCase("keywords")) {
			insertKeywordset(dbfactory, info, content);
		}

		if (name != null && name.trim().equalsIgnoreCase("subject")) {
			insertSubjectset(dbfactory, info, content);
		}

		return finalizeValues();
	}

	private void insertAuthor(BaseTagFactory dbfactory, ElementImpl info,
			String content) {

		Author author = dbfactory.createAuthor();
		Personname personname = dbfactory.createPersonname();
		author.appendChild(personname);
		personname.setTextContent(content);

		List<Author> authorList = info.findChildren(Author.class);

		if (authorList.size() == 0) {
			info.appendChild(author);
		} else {
			info.insertAfter(author, authorList.get(authorList.size() - 1));
		}
	}

	private void insertDate(BaseTagFactory dbfactory, ElementImpl info,
			String content) {

		Date date = dbfactory.createDate();
		date.setTextContent(content);

		List<Date> dateList = info.findChildren(Date.class);

		if (dateList.size() == 0) {
			info.appendChild(date);
		} else {
			info.insertAfter(date, dateList.get(dateList.size() - 1));
		}
	}

	private void insertAbstract(BaseTagFactory dbfactory, ElementImpl info,
			String content) {

		Abstract description = dbfactory.createAbstract();
		Para para = dbfactory.createPara(content);
		description.appendChild(para);

		List<Abstract> abstractList = info.findChildren(Abstract.class);

		if (abstractList.size() == 0) {
			info.appendChild(description);
		} else {
			info.insertAfter(description,
					abstractList.get(abstractList.size() - 1));
		}
	}

	private void insertKeywordset(BaseTagFactory dbfactory,
			ElementImpl info, String content) {

		if (content == null || content.trim().length() == 0) {
			return;
		}

		Keywordset keywordset = dbfactory.createKeywordset();

		for (String token : content.split(",")) {
			Keyword keyword = dbfactory.createKeyword();
			keyword.setTextContent(token.trim());
			keywordset.appendChild(keyword);
		}

		List<Keywordset> keywordsetList = info.findChildren(Keywordset.class);

		if (keywordsetList.size() == 0) {
			info.appendChild(keywordset);
		} else {
			info.insertAfter(keywordset,
					keywordsetList.get(keywordsetList.size() - 1));
		}
	}

	private void insertSubjectset(BaseTagFactory dbfactory,
			ElementImpl info, String content) {

		if (content == null || content.trim().length() == 0) {
			return;
		}

		List<Subjectset> subjectsetList = info.findChildren(Subjectset.class);

		Subjectset subjectset = null;

		if (subjectsetList.size() == 0) {
			subjectset = dbfactory.createSubjectset();
		} else {
			subjectset = subjectsetList.get(0);
		}

		Subject subject = dbfactory.createSubject();
		subjectset.appendChild(subject);
		Subjectterm subjectterm = dbfactory.createSubjectterm();
		subjectterm.setTextContent(content);
		subject.appendChild(subjectterm);
		info.appendChild(subjectset);
	}
}
