/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/* eslint-disable no-new-wrappers */

'use strict';

// MODULES //

var tape = require( 'tape' );
var hasSymbols = require( './../../has-symbol-support' );
var hasToStringTag = require( './../../has-tostringtag-support' );
var isEmptyString = require( './../lib/object.js' );


// VARIABLES //

var opts = {
	'skip': !hasSymbols()
};


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.strictEqual( typeof isEmptyString, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns `true` if provided an empty string object', function test( t ) {
	t.strictEqual( isEmptyString( new String( '' ) ), true, 'returns true' );
	t.end();
});

tape( 'the function returns `false` if provided a primitive string', function test( t ) {
	t.strictEqual( isEmptyString( '' ), false, 'returns false' );
	t.end();
});

tape( 'if `Symbol.toStringTag` is supported, the function guards against objects masquerading as `String` objects', opts, function test( t ) {
	var mock = {
		'toString': toString,
		'valueOf': valueOf
	};
	if ( hasToStringTag() ) {
		mock[ Symbol.toStringTag ] = 'String';
	}
	t.strictEqual( isEmptyString( mock ), false, 'returns false' );
	t.end();

	function toString() {
		return '';
	}
	function valueOf() {
		return '';
	}
});

tape( 'the function returns `false` if not provided an empty string', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		new String( '5' ),
		5,
		NaN,
		null,
		true,
		void 0,
		[],
		{},
		new Date(),
		/./,
		new RegExp( '.' ),
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.strictEqual( isEmptyString( values[i] ), false, 'returns false when provided '+values[i] );
	}
	t.end();
});
